# datasets/registry.py
from __future__ import annotations

from typing import Callable, Dict, Any

from .base import DatasetSpec, Loader

DATASETS: Dict[str, Loader] = {}

def register(name: str) -> Callable[[Loader], Loader]:
    """Decorator: register a dataset loader by name."""
    def deco(fn: Loader) -> Loader:
        if name in DATASETS:
            raise ValueError(f"Dataset '{name}' already registered")
        DATASETS[name] = fn
        return fn
    return deco

def load(name: str, *, cache_dir: str | None = None, **kwargs: Any) -> DatasetSpec:
    """Load a dataset by registered name."""
    if name not in DATASETS:
        raise KeyError(f"Unknown dataset '{name}'. Known: {sorted(DATASETS)}")
    return DATASETS[name](cache_dir=cache_dir, **kwargs)
